/**
  Door Intruder Detector using ML with the Nicla Voice - Application Note
  Name: Nicla_Voice_Code.ino
  Purpose: Nicla Voice identifying sounds with a ML model built with Edge Impulse to protect and notify through BLE to a Portenta H7 if a door is opened or an intruder has forced it.

  @author Christopher Mendez
*/

#include "NDP.h"
#include <ArduinoBLE.h>

// Alert Service
BLEService alertService("1802");  // Immediate alert

// BLE Alert Characteristic
BLEUnsignedCharCharacteristic alertLevel("2A06",                // standard 16-bit characteristic UUID
                                         BLERead | BLENotify);  // remote clients will be able to get notifications if this characteristic changes

// Bluetooth® Low Energy Battery Level Characteristic
BLEUnsignedCharCharacteristic batteryLevelChar("2A19",                // standard 16-bit characteristic UUID
                                               BLERead | BLENotify);  // remote clients will be able to get notifications if this characteristic changes

// Variable to setup the lowest power consumption for the board if set to true.
const bool lowestPower = false;

// Global Parameters
int oldBatteryLevel = 0;     // last battery level reading.
int firstSent = 0;           // to send just once the BLE characteristics at system boot up.

/*************************************
* Nicla Voice ML node Routines
*************************************/

/**
  Inference Interruption Callback to be executed with every triggered inference,
  it controls the built-in LED's and send the alerts through BLE.
  
  Possible labels: NN0:opened, NN0:forcing

  Alerts: 1 = mild alert (for door opened), 2 = high alert (for intruder detected)

  @param label The infered category label
*/
void BLEsend(char* label) {

  if (strcmp(label, "NN0:opened") == 0) {
    alertLevel.writeValue(1);
    NDP.noInterrupts();
    nicla::leds.begin();
    nicla::leds.setColor(green);
    delay(3000);
    nicla::leds.end();
    NDP.interrupts();
  }
  if (strcmp(label, "NN0:forcing") == 0) {
    alertLevel.writeValue(2);
    NDP.noInterrupts();
    nicla::leds.begin();
    nicla::leds.setColor(red);
    delay(3000);
    nicla::leds.end();
    NDP.interrupts();
  }
  if (!lowestPower) {
    Serial.println(label);
  }
}

/**
  Blinking green LED when called. 
*/
void ledGreenOn() {
  nicla::leds.begin();
  nicla::leds.setColor(green);
  delay(200);
  nicla::leds.setColor(off);
  nicla::leds.end();
}

/**
  Infinite blinking red LED when a system error occurred. 
*/
void ledRedBlink() {
  while (1) {
    nicla::leds.begin();
    nicla::leds.setColor(red);
    delay(200);
    nicla::leds.setColor(off);
    delay(200);
    nicla::leds.end();
  }
}

/**
  Blinking blue LED when called (for BLE connection). 
*/
void ledBlueBlink() {

  for (int i = 0; i <= 2; i++) {
    nicla::leds.begin();
    nicla::leds.setColor(blue);
    delay(200);
    nicla::leds.setColor(off);
    delay(200);
    nicla::leds.end();
  }
}

/**
  Main section setup
*/
void setup() {

  Serial.begin(115200);

  // Nicla System setup
  nicla::begin();
  nicla::disableLDO();
  nicla::enableCharging(100);  // enabling the battery charger
  nicla::leds.begin();

  // Initialize BLE
  if (!BLE.begin()) {
    Serial.println("Starting BLE failed!");
    while (1) {
      
    }
  }

  // BLE service and characteristics setup
  BLE.setLocalName("Nicla Lock");                    // Device Name
  BLE.setAdvertisedService(alertService);            // add the service UUID
  alertService.addCharacteristic(alertLevel);        // add the alert level characteristic
  alertService.addCharacteristic(batteryLevelChar);  // add the alert level characteristic
  BLE.addService(alertService);                      // add the alert service
  alertLevel.writeValue(0);                          // set initial value for this characteristic
  batteryLevelChar.writeValue(0);                    // set initial value for this characteristic

  // Neural Desicion Processor callbacks setup
  NDP.onError(ledRedBlink);
  NDP.onMatch(BLEsend);
  NDP.onEvent(ledGreenOn);

  Serial.println("Loading synpackages");

  // Neural Desicion Processor firmware and ML model files loading
  NDP.begin("mcu_fw_120_v91.synpkg");
  NDP.load("dsp_firmware_v91.synpkg");
  NDP.load("ei_model.synpkg");
  Serial.println("packages loaded");
  NDP.getInfo();
  Serial.println("Configure mic");
  NDP.turnOnMicrophone();


  // start advertising
  BLE.advertise();

  nicla::leds.end();

  // For maximum low power; please note that it's impossible to print after calling these functions
  if (lowestPower) {
    NRF_UART0->ENABLE = 0;
  }
  //NDP.turnOffMicrophone();
}

void loop() {

  BLEDevice central = BLE.central();

  if (central) {
    // start inferencing after BLE connected
    ledBlueBlink();
    NDP.interrupts();

    while (central.connected()) {

      // Battery charger output logic to record the actual battery level
      int batteryLevel = nicla::getBatteryVoltagePercentage();

      // send the battery status just once after connected to central
      if (firstSent == 0) {
        batteryLevelChar.writeValue(batteryLevel);
        Serial.println("First battery status sent");
        Serial.println(batteryLevel);
        firstSent = 1;
      }

      if (batteryLevel != oldBatteryLevel && firstSent) {  // if the battery level has changed
        batteryLevelChar.writeValue(batteryLevel);         // update the battery level characteristic
        oldBatteryLevel = batteryLevel;                    // save the level for next comparison
      }

      /*if(firstSent){
        batteryLevelChar.writeValue(batteryLevel);         // update the battery level characteristic
      }*/

      // sleep and save power
      delay(1000);
    }

  } else {
    // stop inferencing after BLE disconnected
    NDP.noInterrupts();
    alertLevel.writeValue(0);
    firstSent = 0;
    delay(1000);
  }
}